% 3D Time Reversal Reconstruction For A Planar Sensor Example
%
% This example demonstrates the use of k-Wave for the reconstruction of a
% three-dimensional photoacoustic wave-field recorded over a planar sensor
% array. The sensor data is simulated and then time-reversed using
% kspaceFirstOrder3D. It builds on the 3D FFT Reconstruction For A Planar
% Sensor and 2D Time Reversal Reconstruction For A Line Sensor examples. 
%
% author: Bradley Treeby
% date: 10th July 2009
% last update: 16th July 2009
%  
% This example is part of the k-Wave Toolbox (http://www.k-wave.org)

clear all;

% =========================================================================
% SIMULATION
% =========================================================================

% change scale to 2 to reproduce the higher resolution figures given in the
% help menu
scale = 1;

% create the computational grid
PML_size = 8;       % size of the PML in pixels
Nx = 64*scale;      % number of pixels in the x direction
Ny = 64*scale;      % number of pixels in the y direction
Nz = 32*scale;      % number of pixels in the z direction
dx = 0.2e-3/scale;  % pixel width [m]
dy = 0.2e-3/scale;  % pixel width [m]
dz = 0.2e-3/scale;  % pixel height [m]
kgrid = makeGrid(Nx, dx, Ny, dy, Nz, dz);

% define the properties of the propagation medium
c = 1500;           % [m/s]
rho = 1000;         % [kg/m^3]

% create initial pressure distribution using makeBall
ball_magnitude = 10;
ball_x_pos = 36*scale;    % pixels
ball_y_pos = 36*scale;    % pixels
ball_z_pos = 16*scale;    % pixels
ball_radius = 3*scale;    % pixels
p0 = ball_magnitude*makeBall(Nx, Ny, Nz, ball_x_pos, ball_y_pos, ball_z_pos, ball_radius);

% smooth the initial pressure distribution and restore the magnitude
p0 = smooth(p0, kgrid, true);

% define a binary planar sensor
sensor_mask = zeros(kgrid.Nz, kgrid.Nx, kgrid.Ny);
sensor_mask(1+PML_size, 1+PML_size:end-PML_size, 1+PML_size:end-PML_size) = 1;

% create the time array
[t_array dt] = makeTime(kgrid, c);

% set the input arguements
input_args = {'PMLSize', PML_size, 'Smooth', false};

% run the simulation
sensor_data = kspaceFirstOrder3D(p0, kgrid, c, rho, t_array, sensor_mask, input_args{:});

% add time reversal flag to the input options
input_args = [input_args, {'TimeRev', true}];

% run the time-reversal reconstruction
p0_recon = kspaceFirstOrder3D(sensor_data, kgrid, c, rho, t_array, sensor_mask, input_args{:});

% add first order compensation for only recording over a half plane
p0_recon = p0_recon*2;

% apply a positivity condition
p0_recon(p0_recon < 0) = 0;

% =========================================================================
% VISUALISATION
% =========================================================================

% plot the initial pressure
figure;
plot_scale = [-10 10];
subplot(2, 2, 1), imagesc(kgrid.x(1,:,1)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0(:, :, ball_y_pos)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid.y(1,1,:)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0(:, ball_x_pos, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid.x(1,:,1)*1e3, kgrid.y(1,1,:)*1e3, squeeze(p0(ball_z_pos, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);

% plot the reconstructed initial pressure
figure;
subplot(2, 2, 1), imagesc(kgrid.x(1,:,1)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon(:, :, ball_y_pos)), plot_scale);
title('x-z plane');
axis image;
subplot(2, 2, 2), imagesc(kgrid.y(1,1,:)*1e3, kgrid.z(:,1,1)*1e3, squeeze(p0_recon(:, ball_x_pos, :)), plot_scale);
title('y-z plane');
axis image;
xlabel('(All axes in mm)');
subplot(2, 2, 3), imagesc(kgrid.x(1,:,1)*1e3, kgrid.y(1,1,:)*1e3, squeeze(p0_recon(ball_z_pos, :, :)).', plot_scale);
title('x-y plane');
axis image;
colormap(getColorMap);